<?php

namespace App\Http\Controllers;

use App\Blog;
use Illuminate\Http\Request;
use App\Http\Requests\ValidationRequest;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Helper;
use File;
use Image;
use Alert;
use Illuminate\Database\QueryException;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Storage;
use PDF;
use Yajra\DataTables\Facades\DataTables;

class BlogController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        return view('pages.be.blogs');
    }

    function json()
    {
        set_time_limit(0);
        ini_set('memory_limit', '-1');
        $data = Blog::orderBy('created_at', 'DESC')->get();
        return DataTables::of($data)
                    ->addIndexColumn()
                    ->editColumn('options', function($row){
                        $edit = route('blog.edit', $row->id);
                        $downloadPDF = route('blog.downloadPDF', $row->slug);
                        $btn = "&emsp;<a href='{$downloadPDF}' id='downloadPDF' class='btn btn-sm btn-info' title='Download PDF'><i class='fa fa-download'></i></a> <a href='{$edit}' class='btn btn-sm btn-warning modal-show edit' title='Form Edit Blog: {$row->title}'><i class='fa fa-pencil'></i></a> <button type='button' class='btn btn-sm btn-danger' id='destroy' data-id='".$row->id."'><i class='fa fa-trash'></i></button>";
                        return $btn;
                    })
                    ->editColumn('picture', function($row){
                        if($row->picture != NULL){
                            $picture = "<img src=".url('assets/files/blog/'.$row->picture.'?time='.time())." width='100px'>";
                        }else{
                            $picture = "<img src=".url('assets/frontend/img/no-picture.png?time='.time())." width='100px'>";
                        }
                        return $picture;
                    })
                    ->editColumn('status', function($row){
                        if($row->status == 1){
                            $status = "<label class='switch switch-primary'><input data-id='".$row->id."' type='checkbox' id='status' checked ><span></span></label>";
                        }else{
                            $status = "<label class='switch switch-primary'><input data-id='".$row->id."' type='checkbox' id='status' type='checkbox'><span></span></label>";
                        }
                        return $status;
                    })
                    ->editColumn('created_at', function($row){
                        return $row->created_at != NULL ? \Helper::tgl_indo($row->created_at) : "-";
                    })
                    ->editColumn('updated_at', function($row){
                        return $row->updated_at != NULL ? \Helper::tgl_indo($row->updated_at) : "-";
                    })
                    ->rawColumns(['options', 'picture', 'status'])
                    ->toJson();
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $model = new Blog();
        return view('pages.be.form_blog', compact('model'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $this->validate($request, [
                'title' => 'required|string',
                'content' => 'required'
            ],[
                'title.required' => 'Title Konten harus diisi!',
                'title.string' => 'Title Konten hanya string!',
                'content.required' => 'Konten harus diisi!'
            ]);

            $data = new Blog();
            $data->title = $request->title;
            $data->content = $request->content;
            $data->slug = Helper::createBlogSlug($request->title);
            $data->author = \Auth::user()->username;
            $data->created_at = Carbon::now('Asia/Jakarta');
            $data->updated_at = null;
            $data->save();
            \Helper::log(\Auth::user()->username, "Menambah konten blog : ".$data->title);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menambah konten blog!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function show(Blog $blog)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $model = Blog::findOrFail($id);
        return view('pages.be.form_blog', compact('model'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            $this->validate($request, [
                'title' => 'required|string',
                'content' => 'required',
                'updated_at' => 'required|date_format:Y-m-d H:i:s',
            ],[
                'title.required' => 'Title Konten harus diisi!',
                'title.string' => 'Title Konten hanya string!',
                'content.required' => 'Deskripsi harus diisi!',
                'updated_at.required' => 'Tanggal update posting tidak boleh kosong!',
                'date_format' => 'Format tanggal salah!',
            ]);

            $data = array(
                'title' => $request->title,
                'slug' => Helper::createBlogSlug($request->title, $id),
                'content' => $request->content,
                'updated_at' => $request->updated_at,
            );
            $blog = Blog::findOrFail($id);
            $blog->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate konten blog : ".$blog->title);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil update konten!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }

    }

    public function update_status(Request $request){
        try {
            $id = $request->id;
            $cek = Blog::where('id', $id)->first();
            if($cek->status == 0){
            $data = array('status' => '1', 'updated_at' => Carbon::now('Asia/Jakarta'));
            }else{
                $data = array('status' => '0', 'updated_at' => Carbon::now('Asia/Jakarta'));
            }
            $blog = Blog::findOrFail($id);
            $blog->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate status konten blog : ".$blog->title);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengubah status konten!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    public function update_picture(ValidationRequest $request)
    {
        try {
            $this->validate($request, [
                'picture' => 'required'
            ],[
            'picture.required' => 'Gambar tidak boleh kosong!'
            ]);
            $id = $request->id;
            $blog = Blog::findOrFail($id);

            if(file_exists("assets/files/blog/".$blog->picture)){
                File::delete('assets/files/blog/'.$blog->picture);
                if(file_exists("assets/files/blog/thumbnail/".$blog->picture)){
                    File::delete('assets/files/blog/thumbnail/'.$blog->picture);
                }
            }

            if(!File::isDirectory('assets/files/blog')){
                File::makeDirectory('assets/files/blog', 0755, true, true);
            }
            if(!File::isDirectory('assets/files/blog/thumbnail')) {
                File::makeDirectory('assets/files/blog/thumbnail', 0755, true, true);
            }

            $file = $request->file('picture');
            $filename = strtolower('gambar-'.Helper::acakstring(8)).'.'.$request->picture->extension();
            $path = "assets/files/blog/".$filename;
            $uploaded = Helper::cropImage($file, $path, 800, 600);
            if($uploaded){
                $thumbnailsource = "assets/files/blog/".$filename;
                $thumbnailpath = "assets/files/blog/thumbnail/". $filename;
                Helper::createThumbnail($thumbnailsource, $thumbnailpath, 300, 225);
            }
            //create backupdata
            // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/blog/", $path, $filename);
            // Storage::disk('SFTP')->putFileAs(Helper::web_config('npsn')."/assets/files/blog/thumbnail", $thumbnailpath, $filename);
            $data = array('picture' => $filename, 'updated_at' => Carbon::now('Asia/Jakarta'));
            $blog->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate gambar konten blog : ".$blog->title);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengganti gambar konten!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    public function hapus_picture(Request $request)
    {
        try {
            $id = $request->id;
            $blog = Blog::findOrFail($id);
            if(file_exists("assets/files/blog/".$blog->picture)){
                File::delete('assets/files/blog/'.$blog->picture);
                if(file_exists("assets/files/blog/thumbnail/".$blog->picture)){
                    File::delete('assets/files/blog/thumbnail/'.$blog->picture);
                }
            }
            $blog->update(['picture' => null, 'updated_at' => Carbon::now('Asia/Jakarta')]);
            \Helper::log(\Auth::user()->username, "Menghapus gambar konten blog : ".$blog->title);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menghapus gambar konten!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Blog  $blog
     * @return \Illuminate\Http\Response
     */
    public function destroy(Blog $blog)
    {
        //
    }

    public function remove(Request $request)
    {
        try {
            $id = $request->input('id');
            $blog = Blog::findOrFail($id);
            $delete = $blog->delete();
            if($delete){
                if(file_exists("assets/files/blog/".$blog->picture)){
                    File::delete('assets/files/blog/'.$blog->picture);
                    if(file_exists("assets/files/blog/thumbnail/".$blog->picture)){
                        File::delete('assets/files/blog/thumbnail/'.$blog->picture);
                    }
                }
            }
            \Helper::log(\Auth::user()->username, "Menghapus konten blog : ".$blog->title);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menghapus konten!']);
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }

    public function downloadPDF($slug)
    {
        set_time_limit(0);
        ini_set('memory_limit', '-1');
        try {
            $data = Blog::where('slug', $slug)->first();
            $opciones_ssl=array(
                "ssl"=>array(
                    "verify_peer"=>false,
                    "verify_peer_name"=>false,
                ),
            );
            if($data->picture != NULL){
                $img_path = 'assets/files/blog/'.$data->picture;
            }else{
                $img_path = 'assets/frontend/img/no-picture.png';
            }
            $extencion = pathinfo($img_path, PATHINFO_EXTENSION);
            $dataImg = file_get_contents($img_path, false, stream_context_create($opciones_ssl));
            $img_base_64 = base64_encode($dataImg);
            $path_img = 'data:image/' . $extencion . ';base64,' . $img_base_64;
            $pdf = PDF::loadView('pages.be.export.pdf', array('data' =>  $data, 'path_img' => $path_img))->setPaper('a4', 'portrait');
            \Helper::doBackup();
            \Helper::log(\Auth::user()->username, "Mendownload konten blog : ".$data->title);
            return $pdf->download($slug.'.pdf');
        } catch (QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => $e->errorInfo]);
        }
    }
}
